/***************************************************************** 
*  @file CRYS_HOST_TST_AES.h
*  @brief Functions used for HOST-SEP communication to send message from HOST to SEP
*         that contains requested API to be executed in SEP and API parameters
*         Used for SEP External application ATP testing.
*
*  @author mottig
*  @remarks Copyright (C) 2004 by Discretix Technologies Ltd.
*           All Rights reserved
*/


/********************** Include Files *************************/

/* .............. CRYS level includes ................. */

#include "DX_VOS_Mem.h"
#include "DX_VOS_Memmap.h"
#include "DX_VOS_Sem.h"
#include "SEPDriver.h" 
#include "ExtApp_qa_op_code.h"
#include "gen.h"
#include "error.h"
#include "CRYS_AES_error.h"
#include "CRYS_HOST_TST_AES.h"
#include "ExtApp_CRYS_Defines.h"


/************************ Defines ******************************/
/************************ MACROS *******************************/
/************************ Global Data **************************/
/**************** Private function prototype *******************/




/************************ Public Functions ******************************/

 /*****************************************************************************
 * Function Name:                                                           
 *  SEPQA_AES_NotIntegrated      
 *  
 * Inputs:
 * IVCounter_ptr - this parameter is the buffer of the IV or counters on mode CTR.
 *                          In ECB mode this parameter has no use.
 *                          In CBC and MAC modes this parameter should containe the IV values.
 *                          in CTR mode this value shuld contain the init counter.
 *
 * Key_ptr - a pointer to the users key buffer.
 *
 * KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
 *
 * EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
 *                           Decrypt operation [1].
 *
 * OperationMode - The operation mode : ECB , CBC , MAC or CTR.
 *
 * is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
 *
 * DataIn_ptr - The pointer to the buffer of the input data to the AES. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
 * ExpectedData_ptr - The pointer to the expected data of the AES operation. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataExpSize - The size of the expected data
 * OutputLocation - indication if input buffer == output buffer or not.
 *
 * FailureFunctionNameOut_ptr - In case that function failures this parameter
 *                                             returns the name of the function that failures
 * FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
 
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function is used to operate the qa external application on the SEP, 
 *  that executes AES not integrated test
 * Algorithm:       
 * 1. initiate and start the message 
 * 2. prepare message with all non buffer parameters of the API
 * 3. send all buffers params                                   
 * 4. end message
 * 5. wait for the response of SEP side
 * 6. start reading message from the SEP  and check returned function status from SEP
 ******************************************************************************/    
EXPORT_C DxUint32 SEPQA_AES_NotIntegrated(
                                    CRYS_AES_IvCounter_t       IVCounter_ptr,       
									CRYS_AES_Key_t             Key_ptr,
									CRYS_AES_KeySize_t         KeySize,            
									CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
									CRYS_AES_OperationMode_t   OperationMode ,      
									DxUint8_t                  Is_secret_key,       
									DxUint8_t                  *DataIn_ptr,        
									DxUint32_t                 DataInSize,
									DxUint8_t                  *ExpectedData_ptr,        
									DxUint32_t                 DataExpSize,
									int						   OutputLocation,
									char                      *FailureFunctionNameOut_ptr,        
									DxUint32_t                 FuncNameMaxSize)
{
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  

   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;
   
   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   
   /* opcode of API function */
   DxUint32_t   Host_opCode = 0;
   	 
   /* key size in bytes */
   DxUint32_t   Host_keySizeInBytes = 0; 
   
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_AES_NotIntegrated_TEST_OP_CODE;

  
   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   Host_messageParam[DX_SEP_HOST_AES_KEYTYPE_OFFSET]        = KeySize;
   Host_messageParam[DX_SEP_HOST_AES_ENCDEC_OFFSET]         = EncryptDecryptFlag;
   Host_messageParam[DX_SEP_HOST_AES_MODE_OFFSET]           = OperationMode;
   Host_messageParam[DX_SEP_HOST_AES_ISSECRETKEY_OFFSET]    = Is_secret_key;
   Host_messageParam[DX_SEP_HOST_AES_OUTPUTLOCATION_OFFSET] = OutputLocation;
   Host_messageParam[DX_SEP_HOST_AES_DATAINSIZE_OFFSET]     = DataInSize;
   Host_messageParam[DX_SEP_HOST_AES_DATAEXPSIZE_OFFSET]    = DataExpSize;
   
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            &Host_sramOffset,
                            DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send IV counter */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)IVCounter_ptr , 
                                     sizeof(CRYS_AES_IvCounter_t) ,
                                     AES_MaxIVCounterSize * 4,
                                     &Host_sramOffset , 
                                     DX_TRUE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send key */
   switch(KeySize)
   {
    case CRYS_AES_Key128BitSize:
      Host_keySizeInBytes = 16;
      break;
      
    case CRYS_AES_Key192BitSize:
      Host_keySizeInBytes = 24;
      break;
      
    case CRYS_AES_Key256BitSize:
      Host_keySizeInBytes = 32;
      break;
      
    default:
      Host_error = CRYS_AES_ILLEGAL_KEY_SIZE_ERROR;
      goto end_function_unlock;
      
   }
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Key_ptr , 
                                     Host_keySizeInBytes,
                                     AES_MaxKeySize*4, 
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
                            
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send data in */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataExpSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send expected data */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr , 
                                     DataExpSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* 4. end message */
   /******************/
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* 5. wait for the response of SEP side */
   /****************************************/
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   
   /* 6. start reading message from the SEP  and check returned function status from SEP */
   /****************************************************************************/
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;
   
   
}/* END OF SEPQA_AES_NotIntegrated */


/*****************************************************************************
 * Function Name:                                                           
 *  SEPQA_AES_Integrated      
 *  
 * Inputs:
 * IVCounter_ptr - this parameter is the buffer of the IV or counters on mode CTR.
 *                          In ECB mode this parameter has no use.
 *                          In CBC and MAC modes this parameter should containe the IV values.
 *                          in CTR mode this value shuld contain the init counter.
 *
 * Key_ptr - a pointer to the users key buffer.
 *
 * KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
 *
 * EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
 *                           Decrypt operation [1].
 *
 * OperationMode - The operation mode : ECB , CBC , MAC or CTR.
 *
 * is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
 *
 * DataIn_ptr - The pointer to the buffer of the input data to the AES. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
 * ExpectedData_ptr - The pointer to the expected data of the AES operation. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataExpSize - The size of the expected data
 * OutputLocation - indication if input buffer == output buffer or not.
 *
 * FailureFunctionNameOut_ptr - In case that function failures this parameter
 *                                             returns the name of the function that failures
 * FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
 
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function is used to operate the qa external application on the SEP, 
 *  that executes AES not integrated test
 * Algorithm:       
 * 1. initiate and start the message 
 * 2. prepare message with all non buffer parameters of the API
 * 3. send all buffers params                                   
 * 4. end message
 * 5. wait for the response of SEP side
 * 6. start reading message from the SEP  and check returned function status from SEP
 ******************************************************************************/    
 EXPORT_C DxUint32 SEPQA_AES_Integrated(
								CRYS_AES_IvCounter_t       IVCounter_ptr,       
								CRYS_AES_Key_t             Key_ptr,
								CRYS_AES_KeySize_t         KeySize,            
								CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
								CRYS_AES_OperationMode_t   OperationMode ,      
								DxUint8_t                  Is_secret_key,       
								DxUint8_t                  *DataIn_ptr,        
								DxUint32_t                 DataInSize,
								DxUint8_t                  *ExpectedData_ptr,        
								DxUint32_t                 DataExpSize,
								int						   OutputLocation,
								char                      *FailureFunctionNameOut_ptr,        
								DxUint32_t                 FuncNameMaxSize)   
{
    /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  

   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;
   
   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   
   /* opcode of API function */
   DxUint32_t   Host_opCode = 0;
   	 
   /* key size in bytes */
   DxUint32_t   Host_keySizeInBytes = 0; 
   
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_AES_Integrated_TEST_OP_CODE;

  
   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   Host_messageParam[DX_SEP_HOST_AES_KEYTYPE_OFFSET]        = KeySize;
   Host_messageParam[DX_SEP_HOST_AES_ENCDEC_OFFSET]         = EncryptDecryptFlag;
   Host_messageParam[DX_SEP_HOST_AES_MODE_OFFSET]           = OperationMode;
   Host_messageParam[DX_SEP_HOST_AES_ISSECRETKEY_OFFSET]    = Is_secret_key;
   Host_messageParam[DX_SEP_HOST_AES_OUTPUTLOCATION_OFFSET] = OutputLocation;
   Host_messageParam[DX_SEP_HOST_AES_DATAINSIZE_OFFSET]     = DataInSize;
   Host_messageParam[DX_SEP_HOST_AES_DATAEXPSIZE_OFFSET]    = DataExpSize;
   
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            &Host_sramOffset,
                            DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send IV counter */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)IVCounter_ptr , 
                                     sizeof(CRYS_AES_IvCounter_t) ,
                                     AES_MaxIVCounterSize * 4,
                                     &Host_sramOffset , 
                                     DX_TRUE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send key */
   switch(KeySize)
   {
    case CRYS_AES_Key128BitSize:
      Host_keySizeInBytes = 16;
      break;
      
    case CRYS_AES_Key192BitSize:
      Host_keySizeInBytes = 24;
      break;
      
    case CRYS_AES_Key256BitSize:
      Host_keySizeInBytes = 32;
      break;
      
    default:
      Host_error = CRYS_AES_ILLEGAL_KEY_SIZE_ERROR;
      goto end_function_unlock;
      
   }
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Key_ptr , 
                                     Host_keySizeInBytes,
                                     AES_MaxKeySize*4, 
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
                            
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send data in */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataExpSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send expected data */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr , 
                                     DataExpSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* 4. end message */
   /******************/
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* 5. wait for the response of SEP side */
   /****************************************/
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   
   /* 6. start reading message from the SEP  and check returned function status from SEP */
   /****************************************************************************/
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;
       
} /* END OF SEPQA_AES_Integrated */



/*****************************************************************************
 * Function Name:                                                           
 *  SEPQA_AES_MAC_NON_Integrated      
 *  
 * Inputs:
 * IVCounter_ptr - this parameter is the buffer of the IV or counters on mode CTR.
 *                          In ECB mode this parameter has no use.
 *                          In CBC and MAC modes this parameter should containe the IV values.
 *                          in CTR mode this value shuld contain the init counter.
 *
 * Key_ptr - a pointer to the users key buffer.
 *
 * KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
 *
 * EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
 *                           Decrypt operation [1].
 *
 * OperationMode - The operation mode : ECB , CBC , MAC or CTR.
 *
 * is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
 *
 * DataIn_ptr - The pointer to the buffer of the input data to the AES. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
 * ExpectedData_ptr - The pointer to the expected data of the AES operation. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataExpSize - The size of the expected data
 * OutputLocation - indication if input buffer == output buffer or not.
 *
 * FailureFunctionNameOut_ptr - In case that function failures this parameter
 *                                             returns the name of the function that failures
 * FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
 
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function is used to operate the qa external application on the SEP, 
 *  that executes AES not integrated test
 * Algorithm:       
 * 1. initiate and start the message 
 * 2. prepare message with all non buffer parameters of the API
 * 3. send all buffers params                                   
 * 4. end message
 * 5. wait for the response of SEP side
 * 6. start reading message from the SEP  and check returned function status from SEP
 ******************************************************************************/    
 EXPORT_C DxUint32 SEPQA_AES_MAC_NON_Integrated(
								CRYS_AES_IvCounter_t       IVCounter_ptr,       
								CRYS_AES_Key_t             Key_ptr,
								CRYS_AES_KeySize_t         KeySize,            
								CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
								CRYS_AES_OperationMode_t   OperationMode ,      
								DxUint8_t                  Is_secret_key,       
								DxUint8_t                  *DataIn_ptr,        
								DxUint32_t                 DataInSize,
								DxUint8_t                  *ExpectedData_ptr,        
								DxUint32_t                 DataExpSize,
								int						   OutputLocation,
								char                      *FailureFunctionNameOut_ptr,        
								DxUint32_t                 FuncNameMaxSize)   
{
    /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  

   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;
   
   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   
   /* opcode of API function */
   DxUint32_t   Host_opCode = 0;
   	 
   /* key size in bytes */
   DxUint32_t   Host_keySizeInBytes = 0; 
   
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_AES_MACNonIntegrated_TEST_OP_CODE;

  
   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   Host_messageParam[DX_SEP_HOST_AES_KEYTYPE_OFFSET]        = KeySize;
   Host_messageParam[DX_SEP_HOST_AES_ENCDEC_OFFSET]         = EncryptDecryptFlag;
   Host_messageParam[DX_SEP_HOST_AES_MODE_OFFSET]           = OperationMode;
   Host_messageParam[DX_SEP_HOST_AES_ISSECRETKEY_OFFSET]    = Is_secret_key;
   Host_messageParam[DX_SEP_HOST_AES_OUTPUTLOCATION_OFFSET] = OutputLocation;
   Host_messageParam[DX_SEP_HOST_AES_DATAINSIZE_OFFSET]     = DataInSize;
   Host_messageParam[DX_SEP_HOST_AES_DATAEXPSIZE_OFFSET]    = DataExpSize;
   
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            &Host_sramOffset,
                            DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send IV counter */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)IVCounter_ptr , 
                                     sizeof(CRYS_AES_IvCounter_t) ,
                                     AES_MaxIVCounterSize * 4,
                                     &Host_sramOffset , 
                                     DX_TRUE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send key */
   switch(KeySize)
   {
    case CRYS_AES_Key128BitSize:
      Host_keySizeInBytes = 16;
      break;
      
    case CRYS_AES_Key192BitSize:
      Host_keySizeInBytes = 24;
      break;
      
    case CRYS_AES_Key256BitSize:
      Host_keySizeInBytes = 32;
      break;
      
    default:
      Host_error = CRYS_AES_ILLEGAL_KEY_SIZE_ERROR;
      goto end_function_unlock;
      
   }
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Key_ptr , 
                                     Host_keySizeInBytes,
                                     AES_MaxKeySize*4, 
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
                            
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send data in */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataExpSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send expected data */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr , 
                                     DataExpSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* 4. end message */
   /******************/
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* 5. wait for the response of SEP side */
   /****************************************/
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   
   /* 6. start reading message from the SEP  and check returned function status from SEP */
   /****************************************************************************/
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;
       
} /* END OF SEPQA_AES_MAC_NON_Integrated */

/*****************************************************************************
 * Function Name:                                                                 
 *  SEPQA_AES_XCBC_CMAC_NON_integrated
 *
 * Inputs:
 * 
 *
 * Key_ptr - a pointer to the users key buffer.
 *
 * KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
 *
 * EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
 *                           Decrypt operation [1].
 *
 * OperationMode - The operation mode : ECB , CBC , MAC or CTR.
 *
 * is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
 *
 * DataIn_ptr - The pointer to the buffer of the input data to the AES. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
 * ExpectedData_ptr - The pointer to the expected data of the AES operation. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataExpSize - The size of the expected data
 * OutputLocation - indication if input buffer == output buffer or not.
 *
 * FailureFunctionNameOut_ptr - In case that function failures this parameter
 *                                             returns the name of the function that failures
 * FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
 
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function is used to operate the qa external application on the SEP, 
 *  that executes AES not integrated test
 * Algorithm:       
 * 1. initiate and start the message 
 * 2. prepare message with all non buffer parameters of the API
 * 3. send all buffers params                                   
 * 4. end message
 * 5. wait for the response of SEP side
 * 6. start reading message from the SEP  and check returned function status from SEP
 ******************************************************************************/    
 EXPORT_C DxUint32 SEPQA_AES_XCBC_CMAC_NON_integrated(     
								CRYS_AES_Key_t             Key_ptr,
								CRYS_AES_KeySize_t         KeySize,            
								CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
								CRYS_AES_OperationMode_t   OperationMode ,      
								DxUint8_t                  Is_secret_key,       
								DxUint8_t                  *DataIn_ptr,        
								DxUint32_t                 DataInSize,
								DxUint8_t                  *ExpectedData_ptr,        
								DxUint32_t                 DataExpSize,
								int						   OutputLocation,
								char                      *FailureFunctionNameOut_ptr,        
								DxUint32_t                 FuncNameMaxSize)   
{
    /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  

   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;
   
   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   
   /* opcode of API function */
   DxUint32_t   Host_opCode = 0;
   	 
   /* key size in bytes */
   DxUint32_t   Host_keySizeInBytes = 0; 
   
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   
   CRYS_AES_IvCounter_t	IVCounter = {0};
   	
   
   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_AES_XCBC_CMACNonIntegrated_TEST_OP_CODE;

  
   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   Host_messageParam[DX_SEP_HOST_AES_KEYTYPE_OFFSET]        = KeySize;
   Host_messageParam[DX_SEP_HOST_AES_ENCDEC_OFFSET]         = EncryptDecryptFlag;
   Host_messageParam[DX_SEP_HOST_AES_MODE_OFFSET]           = OperationMode;
   Host_messageParam[DX_SEP_HOST_AES_ISSECRETKEY_OFFSET]    = Is_secret_key;
   Host_messageParam[DX_SEP_HOST_AES_OUTPUTLOCATION_OFFSET] = OutputLocation;
   Host_messageParam[DX_SEP_HOST_AES_DATAINSIZE_OFFSET]     = DataInSize;
   Host_messageParam[DX_SEP_HOST_AES_DATAEXPSIZE_OFFSET]    = DataExpSize;
   
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            &Host_sramOffset,
                            DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send IV counter */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)IVCounter , 
                                     sizeof(CRYS_AES_IvCounter_t) ,
                                     AES_MaxIVCounterSize * 4,
                                     &Host_sramOffset , 
                                     DX_TRUE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send key */
   switch(KeySize)
   {
    case CRYS_AES_Key128BitSize:
      Host_keySizeInBytes = 16;
      break;
      
    case CRYS_AES_Key192BitSize:
      Host_keySizeInBytes = 24;
      break;
      
    case CRYS_AES_Key256BitSize:
      Host_keySizeInBytes = 32;
      break;
      
    default:
      Host_error = CRYS_AES_ILLEGAL_KEY_SIZE_ERROR;
      goto end_function_unlock;
      
   }
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Key_ptr , 
                                     Host_keySizeInBytes,
                                     AES_MaxKeySize*4, 
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
                            
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send data in */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataExpSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send expected data */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr , 
                                     DataExpSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* 4. end message */
   /******************/
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* 5. wait for the response of SEP side */
   /****************************************/
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   
   /* 6. start reading message from the SEP  and check returned function status from SEP */
   /****************************************************************************/
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;
       
} /* END OF SEPQA_AES_XCBC_CMAC_NON_integrated */





/*****************************************************************************
 * Function Name:                                                                 
 *  SEPQA_AES_XCBC_CMAC_Integrated
 *
 * Inputs:
 * 
 *
 * Key_ptr - a pointer to the users key buffer.
 *
 * KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
 *
 * EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
 *                           Decrypt operation [1].
 *
 * OperationMode - The operation mode : ECB , CBC , MAC or CTR.
 *
 * is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
 *
 * DataIn_ptr - The pointer to the buffer of the input data to the AES. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
 * ExpectedData_ptr - The pointer to the expected data of the AES operation. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataExpSize - The size of the expected data
 * OutputLocation - indication if input buffer == output buffer or not.
 *
 * FailureFunctionNameOut_ptr - In case that function failures this parameter
 *                                             returns the name of the function that failures
 * FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
 
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function is used to operate the qa external application on the SEP, 
 *  that executes AES not integrated test
 * Algorithm:       
 * 1. initiate and start the message 
 * 2. prepare message with all non buffer parameters of the API
 * 3. send all buffers params                                   
 * 4. end message
 * 5. wait for the response of SEP side
 * 6. start reading message from the SEP  and check returned function status from SEP
 ******************************************************************************/    
 EXPORT_C DxUint32 SEPQA_AES_XCBC_CMAC_integrated(     
								CRYS_AES_Key_t             Key_ptr,
								CRYS_AES_KeySize_t         KeySize,            
								CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
								CRYS_AES_OperationMode_t   OperationMode ,      
								DxUint8_t                  Is_secret_key,       
								DxUint8_t                  *DataIn_ptr,        
								DxUint32_t                 DataInSize,
								DxUint8_t                  *ExpectedData_ptr,        
								DxUint32_t                 DataExpSize,
								int						   OutputLocation,
								char                      *FailureFunctionNameOut_ptr,        
								DxUint32_t                 FuncNameMaxSize)   
{
    /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  

   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;
   
   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   
   /* opcode of API function */
   DxUint32_t   Host_opCode = 0;
   	 
   /* key size in bytes */
   DxUint32_t   Host_keySizeInBytes = 0; 
   
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   
   CRYS_AES_IvCounter_t	IVCounter = {0};
   	
   
   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_AES_XCBC_CMACIntegrated_TEST_OP_CODE;

  
   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   Host_messageParam[DX_SEP_HOST_AES_KEYTYPE_OFFSET]        = KeySize;
   Host_messageParam[DX_SEP_HOST_AES_ENCDEC_OFFSET]         = EncryptDecryptFlag;
   Host_messageParam[DX_SEP_HOST_AES_MODE_OFFSET]           = OperationMode;
   Host_messageParam[DX_SEP_HOST_AES_ISSECRETKEY_OFFSET]    = Is_secret_key;
   Host_messageParam[DX_SEP_HOST_AES_OUTPUTLOCATION_OFFSET] = OutputLocation;
   Host_messageParam[DX_SEP_HOST_AES_DATAINSIZE_OFFSET]     = DataInSize;
   Host_messageParam[DX_SEP_HOST_AES_DATAEXPSIZE_OFFSET]    = DataExpSize;
   
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            &Host_sramOffset,
                            DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send IV counter */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)IVCounter , 
                                     sizeof(CRYS_AES_IvCounter_t) ,
                                     AES_MaxIVCounterSize * 4,
                                     &Host_sramOffset , 
                                     DX_TRUE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send key */
   switch(KeySize)
   {
    case CRYS_AES_Key128BitSize:
      Host_keySizeInBytes = 16;
      break;
      
    case CRYS_AES_Key192BitSize:
      Host_keySizeInBytes = 24;
      break;
      
    case CRYS_AES_Key256BitSize:
      Host_keySizeInBytes = 32;
      break;
      
    default:
      Host_error = CRYS_AES_ILLEGAL_KEY_SIZE_ERROR;
      goto end_function_unlock;
      
   }
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Key_ptr , 
                                     Host_keySizeInBytes,
                                     AES_MaxKeySize*4, 
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
                            
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send data in */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataExpSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send expected data */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr , 
                                     DataExpSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* 4. end message */
   /******************/
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* 5. wait for the response of SEP side */
   /****************************************/
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   
   /* 6. start reading message from the SEP  and check returned function status from SEP */
   /****************************************************************************/
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;
       
} /* END OF SEPQA_AES_XCBC_CMAC_Integrated */



/*****************************************************************************
 * Function Name:                                                           
 *  SEPQA_AES_MAC_Integrated      
 *  
 * Inputs:
 * IVCounter_ptr - this parameter is the buffer of the IV or counters on mode CTR.
 *                          In ECB mode this parameter has no use.
 *                          In CBC and MAC modes this parameter should containe the IV values.
 *                          in CTR mode this value shuld contain the init counter.
 *
 * Key_ptr - a pointer to the users key buffer.
 *
 * KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
 *
 * EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
 *                           Decrypt operation [1].
 *
 * OperationMode - The operation mode : ECB , CBC , MAC or CTR.
 *
 * is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
 *
 * DataIn_ptr - The pointer to the buffer of the input data to the AES. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
 * ExpectedData_ptr - The pointer to the expected data of the AES operation. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataExpSize - The size of the expected data
 * OutputLocation - indication if input buffer == output buffer or not.
 *
 * FailureFunctionNameOut_ptr - In case that function failures this parameter
 *                                             returns the name of the function that failures
 * FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
 
 *
 * Outputs: 
 *  CRYSError_t - Function error return                                     
 *
 * Description:                                                                 
 *  This function is used to operate the qa external application on the SEP, 
 *  that executes AES not integrated test
 * Algorithm:       
 * 1. initiate and start the message 
 * 2. prepare message with all non buffer parameters of the API
 * 3. send all buffers params                                   
 * 4. end message
 * 5. wait for the response of SEP side
 * 6. start reading message from the SEP  and check returned function status from SEP
 ******************************************************************************/    
 EXPORT_C DxUint32 SEPQA_AES_MAC_Integrated(
								CRYS_AES_IvCounter_t       IVCounter_ptr,       
								CRYS_AES_Key_t             Key_ptr,
								CRYS_AES_KeySize_t         KeySize,            
								CRYS_AES_EncryptMode_t     EncryptDecryptFlag, 
								CRYS_AES_OperationMode_t   OperationMode ,      
								DxUint8_t                  Is_secret_key,       
								DxUint8_t                  *DataIn_ptr,        
								DxUint32_t                 DataInSize,
								DxUint8_t                  *ExpectedData_ptr,        
								DxUint32_t                 DataExpSize,
								int						   OutputLocation,
								char                      *FailureFunctionNameOut_ptr,        
								DxUint32_t                 FuncNameMaxSize)   
{
    /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  

   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;
   
   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   
   /* opcode of API function */
   DxUint32_t   Host_opCode = 0;
   	 
   /* key size in bytes */
   DxUint32_t   Host_keySizeInBytes = 0; 
   
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_AES_MACIntegrated_TEST_OP_CODE;

  
   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   Host_messageParam[DX_SEP_HOST_AES_KEYTYPE_OFFSET]        = KeySize;
   Host_messageParam[DX_SEP_HOST_AES_ENCDEC_OFFSET]         = EncryptDecryptFlag;
   Host_messageParam[DX_SEP_HOST_AES_MODE_OFFSET]           = OperationMode;
   Host_messageParam[DX_SEP_HOST_AES_ISSECRETKEY_OFFSET]    = Is_secret_key;
   Host_messageParam[DX_SEP_HOST_AES_OUTPUTLOCATION_OFFSET] = OutputLocation;
   Host_messageParam[DX_SEP_HOST_AES_DATAINSIZE_OFFSET]     = DataInSize;
   Host_messageParam[DX_SEP_HOST_AES_DATAEXPSIZE_OFFSET]    = DataExpSize;
   
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            &Host_sramOffset,
                            DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send IV counter */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)IVCounter_ptr , 
                                     sizeof(CRYS_AES_IvCounter_t) ,
                                     AES_MaxIVCounterSize * 4,
                                     &Host_sramOffset , 
                                     DX_TRUE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send key */
   switch(KeySize)
   {
    case CRYS_AES_Key128BitSize:
      Host_keySizeInBytes = 16;
      break;
      
    case CRYS_AES_Key192BitSize:
      Host_keySizeInBytes = 24;
      break;
      
    case CRYS_AES_Key256BitSize:
      Host_keySizeInBytes = 32;
      break;
      
    default:
      Host_error = CRYS_AES_ILLEGAL_KEY_SIZE_ERROR;
      goto end_function_unlock;
      
   }
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Key_ptr , 
                                     Host_keySizeInBytes,
                                     AES_MaxKeySize*4, 
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
                            
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send data in */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataExpSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send expected data */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr , 
                                     DataExpSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* 4. end message */
   /******************/
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* 5. wait for the response of SEP side */
   /****************************************/
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   
   /* 6. start reading message from the SEP  and check returned function status from SEP */
   /****************************************************************************/
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;
       
} /* END OF SEPQA_AES_MAC_Integrated */



/*****************************************************************************
 * Function Name:                                                                 
 *  SEPQA_AES_KeyWrapUnwarp
 *
 * Inputs:
 * 
 *
 * Key_ptr - a pointer to the users key buffer.
 *
 * KeySize - The size of the KEY used on the AES : 128 , 196 or 256 bits as defined in the enum.
 *
 * EncryptDecryptFlag - This flag determains if the AES shall perform an Encrypt operation [0] or a
 *                           Decrypt operation [1].
 *
 * OperationMode - The operation mode : ECB , CBC , MAC or CTR.
 *
 * is_secret_key - This paramrter defines do we use a secret key [1] or a key defined from the user [0].    
 *
 * DataIn_ptr - The pointer to the buffer of the input data to the AES. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataInSize - The size of the input data (must be not 0 and must be multiple of 16 bytes).
 * ExpectedData_ptr - The pointer to the expected data of the AES operation. 
 *                   The pointer's value does not need to be word-aligned.
 *
 * DataExpSize - The size of the expected data
 * OutputLocation - indication if input buffer == output buffer or not.
 *
 * FailureFunctionNameOut_ptr - In case that function failures this parameter
 *                                             returns the name of the function that failures
 * FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
 
 *
 * Outputs: 
 *  DxUint32 - Function error return                                     
 *
 * Description:                                                                 
 *  This function is used to operate the qa external application on the SEP, 
 *  that executes AES not integrated test
 * Algorithm:       
 * 1. initiate and start the message 
 * 2. prepare message with all non buffer parameters of the API
 * 3. send all buffers params                                   
 * 4. end message
 * 5. wait for the response of SEP side
 * 6. start reading message from the SEP  and check returned function status from SEP
 ******************************************************************************/    
 EXPORT_C DxUint32 SEPQA_AES_KeyWrapUnwarp( CRYS_AES_Key_t      Key_ptr,
									 CRYS_AES_KeySize_t         KeySize,                 
									 DxUint8_t                  is_secret_key,       
									 DxUint8_t                  *DataIn_ptr,        
									 DxUint32_t                 DataInSize,
									 DxUint8_t*                 ExpectedData_ptr,        
									 DxUint32_t                 WrapdDataSize,
									 char                       *FailureFunctionNameOut_ptr,        
									 DxUint32_t                 FuncNameMaxSize)	
{
    /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  

   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 7;
   
   /* parameters */
   DxUint32_t   Host_messageParam[7] = {0};
   
   
   /* opcode of API function */
   DxUint32_t   Host_opCode = 0;
   	 
   /* key size in bytes */
   DxUint32_t   Host_keySizeInBytes = 0; 
   
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   
   CRYS_AES_IvCounter_t	IVCounter = {0};
   	
   
   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_AES_KeyWrapUnwarp_TEST_OP_CODE;

  
   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   Host_messageParam[DX_SEP_HOST_AES_KEYTYPE_OFFSET]        = KeySize;
   Host_messageParam[DX_SEP_HOST_AES_ENCDEC_OFFSET]         = 0;
   Host_messageParam[DX_SEP_HOST_AES_MODE_OFFSET]           = 0;
   Host_messageParam[DX_SEP_HOST_AES_ISSECRETKEY_OFFSET]    = is_secret_key;
   Host_messageParam[DX_SEP_HOST_AES_OUTPUTLOCATION_OFFSET] = 0;
   Host_messageParam[DX_SEP_HOST_AES_DATAINSIZE_OFFSET]     = DataInSize;
   Host_messageParam[DX_SEP_HOST_AES_DATAEXPSIZE_OFFSET]    = WrapdDataSize;
   
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            sizeof(DxUint32_t)*Host_numOfSimpleParams,
                            &Host_sramOffset,
                            DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
   /* send IV counter */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)IVCounter , 
                                     sizeof(CRYS_AES_IvCounter_t) ,
                                     AES_MaxIVCounterSize * 4,
                                     &Host_sramOffset , 
                                     DX_TRUE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* send key */
   switch(KeySize)
   {
    case CRYS_AES_Key128BitSize:
      Host_keySizeInBytes = 16;
      break;
      
    case CRYS_AES_Key192BitSize:
      Host_keySizeInBytes = 24;
      break;
      
    case CRYS_AES_Key256BitSize:
      Host_keySizeInBytes = 32;
      break;
      
    default:
      Host_error = CRYS_AES_ILLEGAL_KEY_SIZE_ERROR;
      goto end_function_unlock;
      
   }
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)Key_ptr , 
                                     Host_keySizeInBytes,
                                     AES_MaxKeySize*4, 
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
                            
   /* calculate word aligned length of the input data */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send data in */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   
   /* calculate word aligned length of the input data */
   Host_maxLength = ((WrapdDataSize+ 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   /* send expected data */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr , 
                                     WrapdDataSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   
   /* 4. end message */
   /******************/
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* 5. wait for the response of SEP side */
   /****************************************/
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   
   /* 6. start reading message from the SEP  and check returned function status from SEP */
   /****************************************************************************/
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;
       
} /* END OF SEPQA_AES_XCBC_CMAC_Integrated */

